
; Secure UHF Rolling Code transmitter
; Housed in PP43 keyfob
; PIC16LF15323-I/SL
; # device <PIC16LF15323>   
#include <xc.inc>
  
; CONFIG1
  CONFIG  FEXTOSC = OFF         ; External Oscillator mode selection bits (Oscillator not enabled)
  CONFIG  RSTOSC = HFINTPLL     ; x 2 PLL total of 32MHz
  CONFIG  CLKOUTEN = OFF        ; Clock Out Enable bit (CLKOUT function is disabled; i/o or oscillator function on OSC2)
  CONFIG  CSWEN = ON            ; Clock Switch Enable bit (Writing to NOSC and NDIV is allowed)
  CONFIG  FCMEN = OFF           ; Fail-Safe Clock Monitor Enable bit (FSCM timer disabled)

; CONFIG2
  CONFIG  MCLRE = ON            ; Master Clear Enable bit (MCLR pin is Master Clear function)
  CONFIG  PWRTE = OFF           ; Power-up Timer Enable bit (PWRT disabled)
  CONFIG  LPBOREN = OFF         ; Low-Power BOR enable bit (ULPBOR disabled)
  CONFIG  BOREN = OFF           ; Brown-out reset enable bits (Brown-out reset disabled)
  CONFIG  BORV = HI             ; Brown-out Reset Voltage Selection (Brown-out Reset Voltage (VBOR) is set to 2.7V)
  CONFIG  ZCD = OFF             ; Zero-cross detect disable (Zero-cross detect circuit is disabled at POR.)
  CONFIG  PPS1WAY = OFF         ; Peripheral Pin Select one-way control (The PPSLOCK bit can be set and cleared repeatedly by software)
  CONFIG  STVREN = OFF          ; Stack Overflow/Underflow Reset Enable bit (Stack Overflow or Underflow will not cause a reset)

; CONFIG3
  CONFIG  WDTCPS = WDTCPS_31    ; WDT Period Select bits (Divider ratio 1:65536; software control of WDTPS)
  CONFIG  WDTE = OFF            ; WDT operating mode (WDT Disabled, SWDTEN is ignored)
  CONFIG  WDTCWS = WDTCWS_7     ; WDT Window Select bits (window always open (100%); software control; keyed access not required)
  CONFIG  WDTCCS = SC           ; WDT input clock selector (Software Control)

; CONFIG4
  CONFIG  BBSIZE = BB512        ; Boot Block Size Selection bits (512 words boot block size)
  CONFIG  BBEN = OFF            ; Boot Block Enable bit (Boot Block disabled)
  CONFIG  SAFEN = OFF           ; SAF Enable bit (SAF disabled)
  CONFIG  WRTAPP = OFF          ; Application Block Write Protection bit (Application Block not write protected)
  CONFIG  WRTB = OFF            ; Boot Block Write Protection bit (Boot Block not write protected)
  CONFIG  WRTC = OFF            ; Configuration Register Write Protection bit (Configuration Register not write protected)
  CONFIG  WRTSAF = OFF          ; Storage Area Flash Write Protection bit (SAF not write protected)
  CONFIG  LVP = OFF             ; Low Voltage Programming Enable bit (High Voltage on MCLR/Vpp must be used for programming)

; CONFIG5
  CONFIG  CP = OFF               ; UserNVM Program memory code protection bit (UserNVM code protection enabled)

; NOTE!!!!
;  **** MPLAB X IDE V5.45
; Add the line below in the Production/Set project configuration/Customise/pic-as Global Options/Additional Options:
; -Wa,-a -Wl,-pPor_Vec=0h,-pIsr_Vec=4h,-pStorage_Vec=700h

; initial values. These are changed using MUI valuse
	
        PSECT   Storage_Vec,global,class=CODE,delta=2
Storage_Vec:
; SEED
		DW			03F72H		 ; seed ms byte 32 bits
		DW			03F2DH		 ; seed 
		DW			03F9BH		 ; seed 
		DW			03F75H		 ; seed ls byte
; Multiplier
		DW			03F00H		 ; multiplier ms byte 24 bits
		DW			03FA1H		 ; multiplier
		DW			03F8EH		 ; multiplier ls byte
; Incrementer
		DW			03F1DH		 ; incrementer 8 bits
; Randomness
		DW			03FCDH		 ; random byte 5 
		DW			03F17H		 ; random byte 6
; Scramble 
		DW			03F00H		 ; scramble data
; MUI uniqueness of rolling code run flags
		DW			03FAAH		; initially AA, then 55 when run
		
; see also 60H to 6CH for storage in RAM 

; Define variables at RAM memory locations
        
; Math
BARGB0		equ	020H	; random number generator
BARGB1		equ	021H	; random number gen
BARGB2		equ	022H	; random number gen
AARGB0		equ	023H	; random number gen
AARGB1		equ	024H	; random number gen
AARGB2		equ	025H	; random number gen
AARGB3		equ	026H	; random number gen
AARGB4		equ 027H	; random number gen
AARGB5		equ	028H	; random number gen
AARGB6		equ	029H	; random number gen

TEMPB0		equ	02AH	; random gen temp files
TEMPB1		equ	02BH	; random gen temp files
TEMPB2		equ	02CH	; random gen temp files
TEMPB3		equ	02DH	; random gen temp files
LOOPCOUNT	equ	02EH	; loop counter in random gen

; Seed
RANDB0		equ	02FH	; random number seed ms
RANDB1		equ	030H	; random number seed
RANDB2		equ	031H	; random number seed
RANDB3		equ	032H	; random number seed ls
RANDB4		equ	033H	; random number last ms byte
RANDB5		equ	034H	; random number last ls byte

; Multiplier
MULT0		equ	035H	; multiplier ms
MULT1		equ	036H	; multiplier 
MULT2		equ	037H	; multiplier ls

; Increment
INCREMENT	equ	038H	; increment value

; Counters
VALUE		equ	039H	; Value of bit for sending data
VALUE_1		equ	03AH	; delay counter
VALUE_2		equ	03BH	; delay counter	
SCRAMBLE	equ	03CH	; data code scramble
SCRAMBLE1	equ	03DH	; data code scramble working register

; Other
TEMPX		equ	03EH	; temporary register
INTER_COUNT	equ	03FH	; interrupt count
HALF_COUNT	equ	040H	; 500ms counter
QUIETING	equ	041H	; transmitter quieting period for receiver
TEMP		equ	042H	; temporary
SYNC_FLG    equ 043H    ; synchronise flag 
R_STORE     equ 044H    ; random sequence store value
RANDOM_LOOP equ 045H    ; random looping length
RPT_COUNT   equ 046H    ; repeat transmission 
 
; RAM working storage for rolling code parameters stored in Flash H1000 onward 
;(each STOREx is for 14-bits with ms bits at 3F.) 
STORE0		equ	060H	; storage of seed 32-bit ms
STORE1		equ	061H	; storage of seed 32-bit
STORE2		equ	062H	; storage of seed 32-bit
STORE3		equ	063H	; storage of seed 32-bit ls

STORE4		equ	064H	; storage of multiplier 24-bit ms
STORE5		equ	065H	; storage of multiplier 24-bit
STORE6		equ	066H	; storage of multiplier 24-bit ls

STORE7		equ	067H	; storage of incrementer 8-bit
STORE8		equ	068H	; byte 5 of random code
STORE9		equ	069H	; byte 6 of random code
STORE10		equ	06AH	; scramble data code
STORE11		equ	06BH	; whether MUI read for uniqueness has run (starts at AA and replaced with STATUS when run)
        
STO_RANDB0  equ 06CH   ; for comparison after randomisation check for the same value as previous
STO_RANDB1  equ 06DH   ; for comparison after randomisation check for the same value as previous
STO_RANDB2  equ 06EH   ; for comparison after randomisation check for the same value as previous
STO_RANDB3  equ 06FH   ; for comparison after randomisation check for the same value as previous
 
; all banks 
IDENTIFIER	equ	070H	; transmit identifier
FLAG		equ	071H	; timing flag	
MUI_H		equ	072H	; unique identifier ms byte used in READ1 subroutine
MUI_L		equ	073H	; unique identifier ls byte
INTCON_STO	equ	074H	; INTCON store (mainly for GIE)		
 
      
; define as required when used     
; INTCON flags
#define GIE 7    
; STATUS flags    
#define C 0
#define Z 2
; FLASH read and write flags
#define NVMREGS 6    
#define RD 0  
#define WR 1
#define FREE 4
#define LWLO 5
#define WREN 2 
; Interrupts
#define IOCIE 4
#define TMR0IE 5
#define TMR0IF 5      
     
; define Power on reset and interrupt vector start addresses
     
; Power-On-Reset entry point
        PSECT   Por_Vec,global,class=CODE,delta=2

Por_Vec:
        goto    MAIN
;
; Interrupt vector
        PSECT   Isr_Vec,global,class=CODE,delta=2
           
Isr_Vec:
INTERRUPT: 
	BANKSEL PIE0
	bcf 	PIE0,IOCIE  ;clear port change interrupt for switches
	BANKSEL	IOCAF
	clrf	IOCAF
	clrf	IOCCF		; clear port change flags
 	BANKSEL	PIR0
	btfss	PIR0,TMR0IF ; if timer flag set continue
	goto    RETN		; return
	bcf     PIR0,TMR0IF ; clear TMRO interrupt flag
	bsf     FLAG,0		; set timing flag at each interrupt (all banks register)
RETN: 
    BANKSEL PORTA
	retfie				; return from interrupt

MAIN:
; oscillator. Set frequency HFINTOSC with 2xPLL set with bits 6,5,4 at 001
    BANKSEL OSCCON1
    movlw   00010011B   ; 
            ; oscillator divider bits 3,2,1,0 set for 4MHz
            ;1001/512 62.5kHz  
            ;1000/256 125kHz
            ;0111/128 250kHz
            ;0110/64 500kHz
            ;0101/32 1MHz  
            ;0100/16 2MHz   
            ;0011/8 4MHz
            ;0010/4 8MHz
            ;0001/2 16MHz
            ;0000/1 32MHz
    movwf   OSCCON1
    
; initialise ports
; PORTA
	BANKSEL PORTA
	clrf 	PORTA 		;Init PORTA
	BANKSEL LATA 		;Data Latch
	clrf 	LATA 
	BANKSEL ANSELA 	
	clrf 	ANSELA 		;digital I/O
	BANKSEL TRISA
	movlw 	00101111B   ;Set RA<0,1,2,3&5 as inputs
	movwf 	TRISA
; pullups
	BANKSEL WPUA
	movwf	WPUA		; inputs set as pullups initially (TRISA values also used for WPU)
; PORTC
	BANKSEL PORTC
	clrf 	PORTC 		;Init PORTC
	BANKSEL LATA 		;Data Latch
	clrf 	LATA 
	BANKSEL ANSELA
	clrf 	ANSELC 		;digital I/O
    clrf    ANSELA
	BANKSEL TRISA
	movlw 	00011001B   ;Set RC0,3&4 as inputs
	movwf 	TRISC
; pullups
	BANKSEL WPUC
	movwf	WPUC		; inputs set with pullups initially (TRISC values also used for WPU) 

; VREGCON not in L version of PIC16LF15323
;	BANKSEL	VREGCON
;	bsf		VREGCON,1   ; VREGPM set to reduce sleep power (code not required with L version)

	BANKSEL PIE0
	bcf		PIE0,IOCIE  ; interrupt on change
	BANKSEL IOCAN
	bsf		IOCAN,5		; porta,4 interrupt on negative
	BANKSEL IOCCN
	bsf		IOCCN,4		; portc,4 interrupt on negative
	bsf		IOCCN,3		; portc,3 interrupt on negative
	BANKSEL IOCAF		; interrupt on change flag portA
	bcf		IOCAF,5
	BANKSEL IOCCF		; interrupt on change flag portC
	bcf		IOCCF,3
	bcf		IOCCF,4		

	BANKSEL T0CON0		; timer0 control 8-bit timer
	movlw	10000000B
	movwf	T0CON0		; enabled 
	BANKSEL T0CON1 	

            ;divider ls bits
            ;0100 = 1:16
            ;0011 = 1:8
            ;0010 = 1:4
            ;0001 = 1:2
            ;0000 = 1:1
    
	movlw	01000010B
	movwf	T0CON1		; /4 for 1024us between bit changes in transmission (with Fosc/4 : 4MHz oscillator)
; timer timeout flag used to get timing in the interrupt
	BANKSEL PORTA		; bank 0
 	call 	DELAY
    clrf    SYNC_FLG    ; synchronise flag 
     
; check if system has been randomised using MUI values
; For Flash memory location H070B. rolling code is not randomised if H'AA'
	call	READ_ALL	; reads flash and places in STORE0-STORE11
	movf	STORE11,w	; RAM STORE11 for H070B, check if randomised already
	xorlw	0AAH
	btfss	STATUS,Z
	goto	CK_ID
; if not H55, randomise
    call	RANDOMISE   ; AAH so do randomisation of rolling code. Then changes to 55H   
    
; read identity inputs to determine identifier
CK_ID:
 	clrf	IDENTIFIER	; convert port inputs to 0-15 ID code 
; test for ID0
	btfss	PORTC,0		; 
	bsf		IDENTIFIER,3
	btfss	PORTA,2		; 
	bsf		IDENTIFIER,2
	btfss	PORTA,1		; 
	bsf		IDENTIFIER,1
	btfss	PORTA,0		; 
	bsf		IDENTIFIER,0

; pullups disable checker (if input is low, then disable pullup)
	BANKSEL	WPUA		; weak pullup portA
	btfsc	IDENTIFIER,0
	bcf		WPUA,0		; pullup off for RA0
	btfsc	IDENTIFIER,1
	bcf 	WPUA,2		; pullup off for RA2
	btfsc	IDENTIFIER,2
	bcf		WPUA,1		; pullup off for RA1
	BANKSEL	WPUC
	btfsc	IDENTIFIER,3
	bcf		WPUC,0		; pullup off for RC0

	BANKSEL PORTA		; bank 0
    
CK_S1:
; if S1 is pressed at power up, set ACK LED (RC5) on (for synchronise to receiver)
    btfsc   PORTA,5
    goto    POWER_DOWN   
    bsf     PORTC,5
    bsf     SYNC_FLG,0  ; synchronise flag 
; wait for S1 open
S1LOOP:
    call    DELAY
    btfss   PORTA,5
    goto    S1LOOP 
    call    DELAY
    btfss   PORTA,5
    goto    S1LOOP    

POWER_DOWN:

	bcf		LATA,4		; UHF Data off
	nop
	nop
	nop
	clrf	PORTC			; RC5,RC2 and RC1 low; power down UHF transmitter and Ack LED
	bcf		INTCON,GIE      ; GIE no interrupts 
	call	DELAY
    
; if sync flag is set (S1 pressed at power up) keep ACK LED on    
    btfss   SYNC_FLG,0  ; synchronise flag 
    goto    RD_SLEEP    ; ready to sleep
    bsf     PORTC,5     ; Ack. LED on
    call	READ_ALL    ; reads flash and stores in STORE0-11
; run new code until switch pressed 
RUN_CODE1:  
    btfss   PORTA,5         ; check when pressed again
    goto    ENDED
    call	ROLL            ; calculate new code 
  
; store values
; update any modified values

	movf	SCRAMBLE,w
	movwf	STORE10

	movf	RANDB0,w
	movwf	STORE0		; seed STORE
	movf	RANDB1,w
	movwf	STORE1		; seed store

	movf	RANDB2,w
	movwf	STORE2		; seed store

	movf	RANDB3,w
	movwf	STORE3		; seed store

	movf	RANDB4,w
	movwf	STORE8		; byte 5

	movf	RANDB5,w
	movwf	STORE9		; byte 6
     
    goto    RUN_CODE1    
ENDED:
    call	FLASH_WRITE
    goto    WAKE
    
RD_SLEEP:    
	BANKSEL T0CON0			; timer0 control
	movlw	00000000B
	movwf	T0CON0			; disabled 

	BANKSEL	PIE0
	bcf		PIE0,TMR0IE     ; clear interrupt enable for TMR0
	bcf		PIR0,TMR0IF     ; clear interrupt flag for TMR0 
	bsf		PIE0,IOCIE      ; set port change interrupt for switches
	BANKSEL	IOCAF
	clrf	IOCAF
	clrf	IOCCF			; clear port change flags
	BANKSEL PORTA           ; bank 0
	sleep					; reduce power and wait until a switch closure

; awakes from sleep
WAKE:    
	call 	DELAY			; switch debounce

	BANKSEL T0CON0			; timer0 control
	movlw	10000000B
	movwf	T0CON0			; enabled 
	BANKSEL	PIE0
	bsf		PIE0,TMR0IE     ; set interrupt enable for TMR0
	bcf		PIR0,TMR0IF     ; clear interrupt flag for TMR0 
	bcf		PIE0,IOCIE      ; clear port change interrupt for switches
	BANKSEL	IOCAF
	clrf	IOCAF
	clrf	IOCCF			; clear port change flags
	BANKSEL PORTA           ; bank 0

; check switch closure
	clrf	TEMP			; switch setting storage
	btfss	PORTA,5			; if low then set bit 1 in TEMP for S1
	bsf		TEMP,1
	btfss	PORTC,4			; S2
	bsf		TEMP,2
	btfss	PORTC,3
	bsf		TEMP,3
	movf	TEMP,w			; record of switch closure
	btfsc	STATUS,Z		; if no switch closed sleep
	goto	POWER_DOWN		; sleep again

    movlw   6               ; minimum of 2
    movwf   RPT_COUNT
NEXT_SEND:
    decfsz  RPT_COUNT,f
    goto    REDO
    goto    STORE_VALUES
REDO:
; Recall seed/code
	movlw	STORE0          ; seed
	call	READ	
	movwf	RANDB0

	movlw	STORE1          ; seed
	call	READ	
	movwf	RANDB1

	movlw	STORE2          ; seed
	call	READ	
	movwf	RANDB2

	movlw	STORE3          ; seed
	call	READ	
	movwf	RANDB3

; Recall multiplier
	movlw	STORE4          ; multiplier
	call	READ	
	movwf	MULT0

	movlw	STORE5          ; multiplier
	call	READ	
	movwf	MULT1

	movlw	STORE6          ; multiplier
	call	READ	
	movwf	MULT2

; Recall increment
	movlw	STORE7          ; increment value
	call	READ	
	movwf	INCREMENT

; Recall byte 5 of code
	movlw	STORE8          ; byte 5
	call	READ	
	movwf	RANDB4

; Recall byte 6 of code
	movlw	STORE9          ; byte 6
	call	READ	
	movwf	RANDB5

; Recall scramble data code
	movlw	STORE10		; 
	call	READ	
	movwf	SCRAMBLE
    
; check for syncronise. S1 closed with ACK LED on  
    btfss   PORTC,5     ; ACK LED if on then synchronise
    goto    RUN_CODE
    btfsc   TEMP,1      ; is it switch S1 
	goto	SYNCHRONISE

; otherwise run code
RUN_CODE:
; Check switches closed reflected to ms bits
	btfsc	TEMP,1      ; S1			; 
	bsf		TEMP,5
	btfsc	TEMP,2      ; S2
	bsf		TEMP,6
	btfsc	TEMP,3      ; S3
	bsf		TEMP,7
; if temp is still 0 then no switch pressed
	movf	TEMP,w
	btfsc	STATUS,Z        ; when zero exit
	goto	POWER_DOWN

; add switch codes to identifier
	movlw	00001111B
	andwf	IDENTIFIER,f	; clear bits 7-4 keep 3-0
	movf	TEMP,w			; switch codes  S3 is bit 7, S2 is bit 5, S1 is bit 6	
	andlw	11100000B		; ensure other bits are 0
	iorwf	IDENTIFIER,f	; switch codes in ms bytes
	bsf		LATC,5			; Acknowledge LED on
	bcf		LATA,4			; data off
	
	movlw	00100110B
	movwf	LATC			; RC1,RC2 high; power up UHF transmitter, Ack LED on RC5
	call    DELAY
 
; Four start bits 1,1,1,1. 
; Send identifier (bits 0,1,2,3 are address)dependent on RA0,1,6,7.
; identifier ms nibble is switch code S1, S2 and S3 are bit 7, 5, 4 respectively 
; send RANDB0,1,2,3,4,5
; Four stop bits 1,0,1,1 (64 bits total)

; send data 

SEND_CODE:
	bsf		LATA,4		; data high signal
	clrf	TMR0L		; start from zero
    movlw   0FFH    
    movwf   TMR0H
	bcf		FLAG,0		; start at 0
    BANKSEL PIR0		; bank 0    
	bcf		PIR0,TMR0IF ; clear interrupt flag
    BANKSEL PORTA		; bank 0
	bsf		INTCON,GIE    ; GIE allow interrupt

; engage transmitter with 50ms high data input 
	movlw	50          ; 50ms
	movwf	QUIETING
READ_FLAG:
	btfss	FLAG,0
	goto	READ_FLAG
	clrf	FLAG
	decfsz	QUIETING,f
	goto	READ_FLAG

; send a 1 for for first of start bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 1,1,1
	bsf		VALUE,0
	call	DRV_BIT		; 1
; wait for x interrupts before sending the 1 bit
	movlw	7           ; interrupts
	movwf	INTER_COUNT
RECYCLE_FLG:
	bsf		VALUE,0
	call	DRV_BIT
	clrf	FLAG
	decfsz	INTER_COUNT,f
	goto	RECYCLE_FLG
	
	bsf		VALUE,0
	call	DRV_BIT		; 1 (16th interrupt)
	bsf		VALUE,0
	call	DRV_BIT		; 1

; send address and switch identifier
	btfsc	IDENTIFIER,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,0
	bsf		VALUE,0	
	call	DRV_BIT

; send data ms bits first.
; send according to scramble value
 
	call	SEND_SCRAMBLE   ; send the data
   
	goto	SEND_STOP

SUB_RANDB0:
	btfsc	RANDB0,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB1:
	btfsc	RANDB1,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB2:
	btfsc	RANDB2,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB3:
	btfsc	RANDB3,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB4:
	btfsc	RANDB4,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB4,0
	bsf		VALUE,0
	call	DRV_BIT
	return
SUB_RANDB5:
	btfsc	RANDB5,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB5,0
	bsf		VALUE,0
	call	DRV_BIT
	return

; send stop bits
; send a 1 for for first of stop bits
SEND_STOP:

	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 0,1,1
	call	DRV_BIT		; 0
	bsf		VALUE,0
	call	DRV_BIT		; 1
	bsf		VALUE,0
	call	DRV_BIT		; 1

LOOP_OFF:	
	bcf		INTCON,GIE  ; GIE stop interrupt
	bcf		LATC,5		; Acknowledge LED off
    
SET_NEW_CODE:
; Roll to new code and store new code.
   
	call	ROLL            ; calculate new code 
    goto    NEXT_SEND
; ========================================================
STORE_VALUES:
	call	READ_ALL        ; reads flash and stores in STORE0-11
; store values
; update any modified values

	movf	SCRAMBLE,w
	movwf	STORE10

	movf	RANDB0,w
	movwf	STORE0		; seed STORE
	
    movf	RANDB1,w
	movwf	STORE1		; seed store

	movf	RANDB2,w
	movwf	STORE2		; seed store

	movf	RANDB3,w
	movwf	STORE3		; seed store

	movf	RANDB4,w
	movwf	STORE8		; byte 5

	movf	RANDB5,w
	movwf	STORE9		; byte 6
     
; then call write to flash 
	call	FLASH_WRITE
	goto	POWER_DOWN	; end of transmission

; *********************************************************************			
SYNCHRONISE:; / REGISTER (sends data to receiver)
; run randomise first with new seed value
;	call	RANDOMISE	; change randomisation with revised seed (stops using registration codes by capture being used to control
; receiver. Unless the correct receiver is also registered with the new codes it won't work if captured code is used.)  
      
	movlw	00000110B
	movwf	LATC		; RC1,2 high; power up UHF transmitter
	call	DELAY	
	bsf		LATA,4		; data high signal
	clrf	TMR0L		; start from zero
    movlw   0FFH    
    movwf   TMR0H
	bcf		FLAG,0		; start at 0

	BANKSEL	PIR0
	bcf		PIR0,TMR0IF ; clear interrupt flag
	bsf		INTCON,GIE    ; GIE allow interrupt
	BANKSEL PORTA		; bank 0	
	bsf		LATC,5		; Acknowledge LED on

; engage transmitter with 50ms high data input 
	movlw	50		; 50ms
	movwf	QUIETING
READ_FLAG0:
	btfss	FLAG,0
	goto	READ_FLAG0
	clrf	FLAG
	decfsz	QUIETING,f
	goto	READ_FLAG0

; sends the identifier, current random RANDB0,1,2,3 seed code, the MULTiplier (MULT0,1,2) code 
; the INCREMENT and scramble (80 bits)
; Four start bits 1,1,1,1. Four stop bits 1,0,1,1 (88 bits total)

; send a 1 for for first of start bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 1,1,1
	bsf		VALUE,0
	call	DRV_BIT		; 1 (interrupt 1)
; wait for x interrupts before sending the 1 bit
	movlw	7           ; number of interrupts
	movwf	INTER_COUNT
RECYCLE_FLG1:
	bsf		VALUE,0
	call	DRV_BIT	
 	clrf	FLAG
	decfsz	INTER_COUNT,f; when 14 interrupts
	goto	RECYCLE_FLG1
	
	bsf		VALUE,0
	call	DRV_BIT		; 1 (interrupt 16)
	bsf		VALUE,0
	call	DRV_BIT		; 1

; send identifier
; send address identifier

	btfsc	IDENTIFIER,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	IDENTIFIER,0
	bsf		VALUE,0
	call	DRV_BIT

; send data ms bits first. Bytes: random seed, multiplier, increment value
; RANDB0 seed value
	btfsc	RANDB0,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB0,0
	bsf		VALUE,0
	call	DRV_BIT
; RANDB1
	btfsc	RANDB1,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB1,0
	bsf		VALUE,0
	call	DRV_BIT
; RANDB2
	btfsc	RANDB2,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB2,0
	bsf		VALUE,0
	call	DRV_BIT
; RANDB3
	btfsc	RANDB3,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	RANDB3,0
	bsf		VALUE,0
	call	DRV_BIT
; end of first block
	
; send stop bits
; send a 1 for for first of stop bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 0,1,1
	call	DRV_BIT		; 0
	bsf		VALUE,0
	call	DRV_BIT		; 1
	bsf		VALUE,0
	call	DRV_BIT		; 1

; start of second block
    call    DELAY
    call    DELAY
    call    DELAY
    call    DELAY

; engage transmitter with 50ms high data input 
	bsf		LATA,4		; data high signal
	movlw	50          ; 50ms
	movwf	QUIETING
READ_FLAGX:
	btfss	FLAG,0
	goto	READ_FLAGX
	clrf	FLAG
	decfsz	QUIETING,f
	goto	READ_FLAGX

; sends the MULTiplier (MULT0,1,2) code 
; the INCREMENT and scramble (80 bits)
; Four start bits 1,1,1,1. Four stop bits 1,0,1,1 (88 bits total)

; send a 1 for for first of start bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. VALUE cleared at DRV_BIT
; then 1,1,1
	bsf		VALUE,0
	call	DRV_BIT		; 1 (interrupt 1)
; wait for x interrupts before sending the 1 bit
	movlw	7   		; No. of interrupts
	movwf	INTER_COUNT
RECYCLE_FLGX:
	bsf		VALUE,0
	call	DRV_BIT	
	clrf	FLAG
	decfsz	INTER_COUNT,f; when 16 interrupts
	goto	RECYCLE_FLGX
	
	bsf		VALUE,0
	call	DRV_BIT		; 1 (interrupt 16)
	bsf		VALUE,0
	call	DRV_BIT		; 1

; multiplier
; MULT0
CONTINUE_WITH_MULT:
	btfsc	MULT0,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT0,0
	bsf		VALUE,0
	call	DRV_BIT
; MULT1
	btfsc	MULT1,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT1,0
	bsf		VALUE,0
	call	DRV_BIT
; MULT2
	btfsc	MULT2,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	MULT2,0
	bsf		VALUE,0
	call	DRV_BIT
; increment value
; INCREMENT
	btfsc	INCREMENT,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	INCREMENT,0
	bsf		VALUE,0
	call	DRV_BIT
; scramble value
; SCRAMBLE
	btfsc	SCRAMBLE,7
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,6
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,5
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,4
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,3
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,2
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,1
	bsf		VALUE,0
	call	DRV_BIT
	btfsc	SCRAMBLE,0
	bsf		VALUE,0
	call	DRV_BIT

; send stop bits
; send a 1 for for first of stop bits
	bsf		VALUE,0
	call	DRV_BIT		; 1. 'VALUE' cleared at DRV_BIT
; then 0,1,1
	call	DRV_BIT		; 0
	bsf		VALUE,0
	call	DRV_BIT		; 1
	bsf		VALUE,0
	call	DRV_BIT		; 1

; data sent 

; ================================================
	call	READ_ALL	; reads flash and stores in STORE0-11
; store values
; update any modified values

	movf	SCRAMBLE,w
	movwf	STORE10

 	movf	RANDB0,w
	movwf	STORE0		; seed STORE
	movf	RANDB1,w
	movwf	STORE1		; seed store

	movf	RANDB2,w
	movwf	STORE2		; seed store

	movf	RANDB3,w
	movwf	STORE3		; seed store

	movf	RANDB4,w
	movwf	STORE8		; byte 5

	movf	RANDB5,w
	movwf	STORE9		; byte 6
  
; then call write to flash 
	call	FLASH_WRITE

; ====================================================
LOOP2:
	bcf		INTCON,GIE    ; GIE stop interrupt
	call	DELAY
	call	DELAY
	call	DELAY
	bcf		LATC,5		; ACK LED off
    clrf    SYNC_FLG    ; synchronise flag off
	goto	POWER_DOWN	; program stopped till reset

; **************************************************************************************

RANDOMISE: ;
; alters the RANDB values, the MULTiplier values and the Increment values depending on MUI values 
; MUI is at 8100h to 8109h 14 bits each
; MUI_H is altered with swapf so it is unlikely to affect ls bits in the MUI_L byte
; Typically MUI_H is 00 so has no effect on MUI_L when the two are xored
    
    bsf		LATC,5		; acknowledge LED on
	call	DELAY		; time to view LED
    call	DELAY
	call	DELAY

; get least significant MUI values MUI0-MUI8 place in consecutive STORE locations
	movlw	0
	call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE0          ; random value RANDB0
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE0,f
    movlw   1
    call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE1          ; random value RANDB1
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE1,f
    movlw	2
	call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE2          ; random value RANDB2
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE2,f
    movlw	3
	call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE3          ; random value RANDB3
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE3,f
    
; if STORE0-3 are all zero (RANDOM numbers for calculation) use original
    movf    STORE0,w
    btfss   STATUS,Z
    goto    R1
    movf    STORE1,w
    btfss   STATUS,Z
    goto    R1
    movf    STORE2,w
    btfss   STATUS,Z
    goto    R1
    movf    STORE3,w
    btfss   STATUS,Z
    goto    R1
; all zero so use original values
    movlw   72h
    movwf   STORE0    
    movlw   2Dh
    movwf   STORE1
    movlw   9Bh
    movwf   STORE2
    movlw   75h
    movwf   STORE3
    
R1:    
    movlw	4
	call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE4          ; multiplier MULT0
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE4,f
    movlw	5
	call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE5          ; multiplier MULT1
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE5,f
    movlw   6
    call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE6          ; multiplier MULT2
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE6,f

; if STORE4-6 are zero (multiplier for calculation) 
    movf    STORE4,w
    btfss   STATUS,Z
    goto    R2
    movf    STORE5,w
    btfss   STATUS,Z
    goto    R2
    movf    STORE6,w
    btfss   STATUS,Z
    goto    R2
   
; all zero so use original values
    movlw   00h
    movwf   STORE4    
    movlw   0A1h
    movwf   STORE5
    movlw   8Eh
    movwf   STORE6
  
R2:    
    movlw	7
	call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE7          ; increment value INCREMENT
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE7,f

; if STORE7 is zero (the increment value for calculation) 
    movf    STORE7,w
    btfss   STATUS,Z
    goto    R3
    movlw   1Dh
    movwf   STORE7    
    
R3:    
    movlw	8
	call	READ2			; values in MUI_H and MUI_L
    movf    MUI_L,w
    movwf   STORE10         ; scramble value SCRAMBLE 
    swapf   MUI_H,w         ; load ms byte
    xorwf   STORE10,f

; ack LED off
	bcf		LATC,2			; Acknowledge LED off
    
 ; store values after randomisation
    movlw   55h         ; randomised flag that it has been done
    movwf   STORE11  
 ; write to flash 
	call	FLASH_WRITE
	return

; .......................................................................

; subroutines

; delay subroutine

DELAY: ; ~100ms
	movlw	100         ; set delay period 
	movwf	VALUE_2		; 
LP_1:
	movlw	250     	; set delay period value 1 
	movwf	VALUE_1		; VALUE_2 = w
LP_2:
    nop
    decfsz	VALUE_1,f
	goto	LP_2
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_1
	return

; subroutine to drive bits high for a 1 and low for a 0 

; VALUE,0 sets the level (0 = low, 1 = high)

DRV_BIT: 				; drive bits
	btfss	FLAG,0		; wait for flag in interrupt to maintain timing sync
	goto	DRV_BIT
	clrf	FLAG
	btfss	VALUE,0
	goto	SEND_LOW
	clrf	VALUE		; zero
; LED
	bsf		LATC,5			; Ack LED on
; Data
	bsf		LATA,4			; data high signal

; cycle high for 500us
	movlw	170
	movwf	HALF_COUNT
LOOP_HIGH:
	decfsz	HALF_COUNT,f
	goto	LOOP_HIGH
	bcf		LATA,4		; low
LOOP_FLG1:
	btfss	FLAG,0
	goto	LOOP_FLG1	; loop till interrupt flag
	return

SEND_LOW:
	bcf		LATC,5		; ack LED off
	bcf		LATA,4		; low signal

; cycle low for 500us
	movlw	170
	movwf	HALF_COUNT
LOOP_LOW:
	decfsz	HALF_COUNT,f
	goto	LOOP_LOW
LOOP_FLG0:
	bsf		LATA,4		; high
	btfss	FLAG,0
	goto	LOOP_FLG0	; loop till interrupt flag
	return

; subroutines

; read data memory
READ:						; 'w' has read data. remove ms nibble 
	andlw	00001111B		; for the STORE0 to STORE11 locations (0060 onward)

; This code block will read 1 word of program
; data will be returned in w for ls byte

	BANKSEL NVMADRL 		; Select Bank for NVMCON registers
	movwf 	NVMADRL 		; Store LSB of address
	movlw 	07H 			; PROG_ADDR_HI ;
	movwf 	NVMADRH 		; Store MSB of address
	bcf 	NVMCON1,NVMREGS ; Do not select Configuration Space
	bsf 	NVMCON1,RD      ; Initiate read
    movf 	NVMDATL,W 		; Get LSB of word
	BANKSEL PORTA           ; bank 0
	return

READ2: ; for reading MUI H8100 to H8109. use 0100 to 0109 for NVMADR
	BANKSEL NVMADRL 		; Select Bank for NVMCON registers
	movwf 	NVMADRL 		; Store LSB of address
	movlw 	01H 			; PROG_ADDR_HI ;
	movwf 	NVMADRH 		; Store MSB of address
	bsf 	NVMCON1,NVMREGS ; select Configuration Space
	bsf 	NVMCON1,RD      ; Initiate read
	movf 	NVMDATL,W 		; Get LSB of word
	movwf	MUI_L
	movf 	NVMDATH,W 		; Get MSB of word
	movwf	MUI_H
	BANKSEL PORTA           ; bank 0
	return

; read flash 1000 to 100B place in store0-store11
READ_ALL:

	movlw	STORE0		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE0
	movlw	STORE1		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE1
	movlw	STORE2		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE2
	movlw	STORE3		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE3
	movlw	STORE4		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE4
	movlw	STORE5		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE5
	movlw	STORE6		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE6
	movlw	STORE7		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE7
	movlw	STORE8		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE8
	movlw	STORE9		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE9
	movlw	STORE10		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE10
	movlw	STORE11		; effective flash ls byte address after removing ms nibble
	call	READ
	movwf	STORE11
	return

FLASH_WRITE:

; Initially, erase the row address
    BANKSEL NVMADRL
	movlw	00H
	movwf 	NVMADRL 		; Load lower 8 bits of erase address boundary
	movlw	07H
	movwf	NVMADRH 		; Load upper 6 bits of erase address boundary
	bcf 	NVMCON1,NVMREGS ; Choose PFM memory area
	bsf		NVMCON1,FREE    ; Specify an erase operation
; read GIE and store for return later
	movf	INTCON,w
	movwf	INTCON_STO
	bcf		INTCON,GIE      ; GIE Disable interrupts during unlock sequence
	call	UNLOCK_SEQ
	bcf		NVMCON1,WREN    ; Disable writes
    
; write routine 
	movlw	07H
	movwf	NVMADRH 		; Load initial flash address
	movlw	00H
	movwf	NVMADRL
	movlw	040H 			; Load initial low byte data address
	movwf	FSR0L		
	movlw	20H             ; high byte address
	movwf	FSR0H
	bcf		NVMCON1,NVMREGS ; Set Program Flash Memory as write location
	bsf		NVMCON1,WREN    ; Enable writes
	bsf		NVMCON1,LWLO    ; Load only write latches
LOOP:
	moviw	FSR0++
	movwf	NVMDATL 		; Load first data byte
	movlw   03FH            ; 
    movwf	NVMDATH 		; Load second data byte
	
	movf	NVMADRL,W
	xorlw	1FH 			; Check if lower bits of address are 00000
	andlw	1FH             ; and if on last of 32 addresses
	btfsc	STATUS,Z 		; Last of 32 words?
	goto	START_WRITE 	; If so, go write latches into memory
	call	UNLOCK_SEQ 		; If not, go load latch
	incf	NVMADRL,F 		; Increment address
	goto	LOOP
START_WRITE:
	bcf		NVMCON1,LWLO    ; Latch writes complete, now write memory
    
	call	UNLOCK_SEQ 		; Perform required unlock sequence
	bcf		NVMCON1,WREN    ; Disable writes

	btfsc	INTCON_STO,7
	bsf		INTCON,GIE      ; re-enable interrupts if was enabled
	BANKSEL PORTA           ; bank 0
	return

UNLOCK_SEQ:
    bsf		NVMCON1,WREN    ; enable writes
	movlw	55H
	movwf	NVMCON2 		; Begin unlock sequence
	movlw	0AAH
	movwf	NVMCON2
	bsf		NVMCON1,WR
	return

; scramble data send

SEND_SCRAMBLE:
	movf	SCRAMBLE,w
	andlw	00011111B       ; max of 32
	movwf	SCRAMBLE1       ; working value
	btfsc	STATUS,Z        ; Z	if zero
	goto	ZERO_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; z if 1	
	goto	ONE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; Z 	
	goto	TWO_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 3	
	goto	THREE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 4	
	goto	FOUR_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 5	
	goto	FIVE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 6	
	goto	SIX_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 7	
	goto	SEVEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 8	
	goto	EIGHT_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 9	
	goto	NINE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 10	
	goto	TEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 11	
	goto	ELEVEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 12	
	goto	TWELVE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 13	
	goto	THIRTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 14	
	goto	FOURTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 15	
	goto	FIVETEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 16	
	goto	SIXTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 17	
	goto	SEVENTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 18
	goto	EIGHTEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 19	
	goto	NINETEEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 20	
	goto	TWENTY_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 21	
	goto	TWENTYONE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 22	
	goto	TWENTYTWO_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 23	
	goto	TWENTYTHREE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 24	
	goto	TWENTYFOUR_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 25	
	goto	TWENTYFIVE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 26	
	goto	TWENTYSIX_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 27	
	goto	TWENTYSEVEN_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 28	
	goto	TWENTYEIGHT_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 29	
	goto	TWENTYNINE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 30	
	goto	THIRTY_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 31	
	goto	THIRTYONE_SCRAMBLE
	decf	SCRAMBLE1,f
	btfsc	STATUS,Z        ; if 32	
	goto	THIRTYTWO_SCRAMBLE

ZERO_SCRAMBLE:
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
ONE_SCRAMBLE:
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWO_SCRAMBLE:
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
THREE_SCRAMBLE:
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
FOUR_SCRAMBLE:
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
FIVE_SCRAMBLE:
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
SIX_SCRAMBLE:
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
SEVEN_SCRAMBLE:
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
EIGHT_SCRAMBLE:
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
NINE_SCRAMBLE:
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TEN_SCRAMBLE:
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
ELEVEN_SCRAMBLE:
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWELVE_SCRAMBLE:
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
THIRTEEN_SCRAMBLE:
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
FOURTEEN_SCRAMBLE:
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
FIVETEEN_SCRAMBLE:
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
SIXTEEN_SCRAMBLE:
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
SEVENTEEN_SCRAMBLE:
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
EIGHTEEN_SCRAMBLE:
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
NINETEEN_SCRAMBLE:
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTY_SCRAMBLE:
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTYONE_SCRAMBLE:
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTYTWO_SCRAMBLE:
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTYTHREE_SCRAMBLE:
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB0
	call	SUB_RANDB4
	call	SUB_RANDB5
	return
TWENTYFOUR_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB3
	return
TWENTYFIVE_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB2
	return
TWENTYSIX_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB1
	call	SUB_RANDB3
	return
TWENTYSEVEN_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB1
	return
TWENTYEIGHT_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB1
	call	SUB_RANDB2
	return
TWENTYNINE_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB0
	call	SUB_RANDB3
	call	SUB_RANDB2
	call	SUB_RANDB1
	return
THIRTY_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB3
	call	SUB_RANDB0
	return
THIRTYONE_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB1
	call	SUB_RANDB2
	call	SUB_RANDB0
	call	SUB_RANDB3
	return
THIRTYTWO_SCRAMBLE:
	call	SUB_RANDB4
	call	SUB_RANDB5
	call	SUB_RANDB1
	call	SUB_RANDB3
	call	SUB_RANDB0
	call	SUB_RANDB2
	return

ROLL:; Random number generator

;	Input:	32 bit initial integer seed in AARGB0, AARGB1, AARGB2, AARGB3

;	Linear congruential random number generator

;	X  <- (a * X(previous) + c) |mod m|

;	X is the result (2**48 bits used) and becomes the seed for next calculation 
;	with multiplier a (24 bit), increment c (8-bit) and
;	modulus m (number of bits of result selected for seed (2**31)-1 

		movf		RANDB0,w		; random numbers
		movwf		AARGB0			; 2**32
        movwf		STO_RANDB0
		movf		RANDB1,w
		movwf		AARGB1
        movwf		STO_RANDB1
		movf		RANDB2,w
		movwf		AARGB2
        movwf		STO_RANDB2
		movf		RANDB3,w
		movwf		STO_RANDB3
		movwf		AARGB3
					
; decrease by 1
		movf		AARGB3,w
		btfsc		STATUS,Z	; if zero decrease next byte
		decf		AARGB2,f
		decf		AARGB3,f

		movf		AARGB2,w
		btfsc		STATUS,Z	; if zero decrease next byte
		decf		AARGB1,f
		decf		AARGB2,f
	
		movf		AARGB1,w
		btfsc		STATUS,Z	; if zero decrease next byte
		decf		AARGB0,f
		decf		AARGB1,f	; modulus 2**32-1	
		bcf			AARGB0,7	; 2**31-1

; increment		
		movf		INCREMENT,w		; increment value
        addwf       AARGB6,F		; c = incrementer
        btfsc       STATUS,Z		; carry to more significant bytes
        incf        AARGB5,F
		btfsc		STATUS,Z
		incf		AARGB4,F
		btfsc		STATUS,Z
		incf		AARGB3,F
		btfsc       STATUS,Z
        incf        AARGB2,F
		btfsc		STATUS,Z
		incf		AARGB1,F
		btfsc		STATUS,Z
		incf		AARGB0,F

; ENTER_MULTipliers
		movf		MULT2,w			; multiplier 
		movwf		BARGB2
		movf		MULT1,w
		movwf		BARGB1
		movf		MULT0,w
		movwf		BARGB0

		call		FXM3224U		; multiply
; transfer values to RAND registers
	
		movf		AARGB5,w
		movwf		RANDB5
		movf		AARGB4,w
		movwf		RANDB4

		movf		AARGB0,w
		movwf		RANDB0
		movf		AARGB1,W
		movwf		RANDB1
		movf		AARGB2,W
		movwf		RANDB2
		movf		AARGB3,W
		movwf		RANDB3
        
 ; if equal to last time (almost never probability), alter values
		
		movf		RANDB0,w
        xorwf       STO_RANDB0,w    
		btfss		STATUS,Z
        goto        RYX
 		movf		RANDB1,w
        xorwf       STO_RANDB1,w    
		btfss		STATUS,Z
        goto        RYX
        movf		RANDB2,w
        xorwf       STO_RANDB2,w    
		btfss		STATUS,Z
        goto        RYX
        movf		RANDB3,w
        xorwf       STO_RANDB3,w    
		btfss		STATUS,Z
        goto        RYX
; all the same value as befor, so make alterations
        movf        MULT0,w
        addwf       RANDB0,f
        movf        MULT1,w
        addwf       RANDB1,f
        movf        MULT2,w
        addwf       RANDB2,f
RYX:
    
; generate further random output with pseudo random sequence generator (31bit)
;(note: Using bit labelling 1-31 (rather than 0 to 30 of traditional binary counts)
; RANDB0,1,2&3 used as the seed    

        movf        SCRAMBLE,w      ; data scramble value (0-255)
        movwf       RANDOM_LOOP
        bsf         STATUS,C        ; set carry 
    RL: ;Return Loop
        rlf         RANDB0,w		; move Q31 to ms bit 
        movwf       R_STORE			; store value for xoring with Q28
; get Q28
        swapf       RANDB0,w		; Q28 to ms bit
; both store and w have required Q31 and Q28 in ms bit
        xorwf       R_STORE,f		; xor Q31 and Q28
        rlf         R_STORE,w       ; move xored value to carry
; move shift register values along 1
        rlf         RANDB3,f		; carry to ls byte	
        rlf         RANDB2,f		; carry to next byte
        rlf         RANDB1,f		; carry to next byte
        rlf         RANDB0,f		; carry to ms byte
        decfsz      RANDOM_LOOP,f   ; loop at scramble value length
        goto        RL              ; return loop
        incf		SCRAMBLE,f      ; increase scramble value each time
        return
        
;            32x24 Bit Unsigned Fixed Point Multiply 32x24 -> 56

;       Input:  32 bit unsigned fixed point multiplicand in AARGB0, AARGB1,
;               AARGB2, AARGB3

;               24 bit unsigned fixed point multiplier in BARGB0, BARGB1,
;               BARGB2

;       Use:    CALL    FXM3224U

;       Output: 56 bit unsigned fixed point product in AARGB0

;       Result: AARG  <--  AARG x BARG

;       Max Timing:     11+617+2 = 630 clocks

;       Min Timing:     11+151 = 162 clocks

;       PM: 11+139+1 = 151              DM: 15

FXM3224U:
               	clrf   		 	AARGB4          ; clear partial product
               	clrf    		AARGB5
               	clrf    		AARGB6
                movf   			AARGB0,W
                movwf   		TEMPB0
                movf   			AARGB1,W
                movwf   		TEMPB1
                movf   			AARGB2,W
                movwf   		TEMPB2
                movf   			AARGB3,W
                movwf   		TEMPB3

               	call 			UMUL3224L

                retlw           00

; UMUL3224L        macro

;       Max Timing:     2+15+6*25+24+2+7*26+25+2+7*27+26 = 617 clks

;       Min Timing:     2+7*6+5+1+7*6+5+1+7*6+5+6 = 151 clks

;       PM: 31+24+2+25+2+26+2+27 = 139            DM: 15

UMUL3224L:      movlw   		8
                movwf   		LOOPCOUNT

LOOPUM3224A:
                rrf    		 	BARGB2,F
                btfsc   		STATUS,C
               	goto   			ALUM3224NAP
                decfsz  		LOOPCOUNT,F
               	goto   			LOOPUM3224A

                movwf   		LOOPCOUNT

LOOPUM3224B:
                rrf     		BARGB1,F
                btfsc   		STATUS,C
               	goto   			BLUM3224NAP
                decfsz  		LOOPCOUNT,F
               	goto   			LOOPUM3224B

                movwf  			LOOPCOUNT

LOOPUM3224C:
                rrf     		BARGB0,F
                btfsc   		STATUS,C
              	goto  	 		CLUM3224NAP
                decfsz  		LOOPCOUNT,F
               	goto  	 		LOOPUM3224C

              	clrf    		AARGB0
               	clrf    		AARGB1
               	clrf    		AARGB2
               	clrf    		AARGB3
                retlw   		00
                
ALUM3224NAP:     bcf     		STATUS,C
               	goto  	 		ALUM3224NA
                
BLUM3224NAP:     bcf     		STATUS,C
               	goto 	  		BLUM3224NA
                
CLUM3224NAP:     bcf     		STATUS,C
              	goto  	 		CLUM3224NA

ALOOPUM3224:
                rrf     		BARGB2,F
                btfss   		STATUS,C
                goto   			ALUM3224NA
                movf    		TEMPB3,W
                addwf   		AARGB3,F
                movf            TEMPB2,W
                btfsc           STATUS,C
                incfsz          TEMPB2,W
                addwf           AARGB2,F
                movf            TEMPB1,W
                btfsc           STATUS,C
                incfsz          TEMPB1,W
                addwf           AARGB1,F
                movf            TEMPB0,W
                btfsc           STATUS,C
                incfsz          TEMPB0,W
                addwf           AARGB0,F

ALUM3224NA:
                rrf    			AARGB0,F
                rrf    			AARGB1,F
                rrf    			AARGB2,F
                rrf             AARGB3,F
                rrf             AARGB4,F
                decfsz  		LOOPCOUNT,F
                goto    		ALOOPUM3224

                movlw   		8
                movwf   		LOOPCOUNT

BLOOPUM3224:
                rrf     		BARGB1,F
                btfss   		STATUS,C
              	goto  	 		BLUM3224NA
                movf    		TEMPB3,W
                addwf   		AARGB3,F
                movf            TEMPB2,W
                btfsc           STATUS,C
                incfsz          TEMPB2,W
                addwf           AARGB2,F
                movf            TEMPB1,W
                btfsc           STATUS,C
                incfsz          TEMPB1,W
                addwf           AARGB1,F
                movf            TEMPB0,W
                btfsc           STATUS,C
                incfsz          TEMPB0,W
                addwf           AARGB0,F

BLUM3224NA:
                rrf    			AARGB0,F
                rrf    			AARGB1,F
                rrf    			AARGB2,F
                rrf             AARGB3,F
                rrf             AARGB4,F
                rrf             AARGB5,F
                decfsz  		LOOPCOUNT,F
              	goto   			BLOOPUM3224

                movlw   		8
                movwf   		LOOPCOUNT

CLOOPUM3224:
                rrf     		BARGB0,F
                btfss   		STATUS,C
               	goto 	  		CLUM3224NA
                movf    		TEMPB3,W
                addwf   		AARGB3,F
                movf            TEMPB2,W
                btfsc           STATUS,C
                incfsz          TEMPB2,W
                addwf           AARGB2,F
                movf            TEMPB1,W
                btfsc           STATUS,C
                incfsz          TEMPB1,W
                addwf           AARGB1,F
                movf            TEMPB0,W
                btfsc           STATUS,C
                incfsz          TEMPB0,W
                addwf           AARGB0,F

CLUM3224NA:
                rrf  			AARGB0,F
                rrf    			AARGB1,F
                rrf    			AARGB2,F
                rrf    			AARGB3,F
                rrf             AARGB4,F
                rrf             AARGB5,F
                rrf             AARGB6,F
                decfsz  		LOOPCOUNT,F
                goto   			CLOOPUM3224
				return	
    	            

	end MAIN